<?php
// 1. 親カテゴリIDを取得する関数（子カテゴリも含む）
function get_all_parent_categories($category_id) {
    $parent_ids = array();
    $parent = get_term($category_id, 'category');
    while ($parent && !is_wp_error($parent) && $parent->parent != 0) {
        $parent_ids[] = (int)$parent->term_id;
        $parent = get_term($parent->parent, 'category');
    }
    return $parent_ids;
}

// 2. 投稿を取得する関数
function get_posts_by_category($category_id) {

    $exclude_categories = get_option('caf_exclude_categories', array());
    if (!is_array($exclude_categories)) $exclude_categories = array();

    $args = array(
        'posts_per_page' => -1,
        'post_type'      => 'post',
        'tax_query'      => array(
            'relation' => 'AND',
            array(
                'taxonomy' => 'category',
                'field'    => 'id',
                'terms'    => array((int)$category_id),
                'operator' => 'IN',
            ),
            array(
                'taxonomy' => 'category',
                'field'    => 'id',
                'terms'    => $exclude_categories,
                'operator' => 'NOT IN',
            ),
        ),
    );

    $query = new WP_Query($args);
    $posts = array();

    if ($query->have_posts()) {
        while ($query->have_posts()) {
            $query->the_post();

            $thumbnail_url = get_the_post_thumbnail_url(get_the_ID(), 'thumbnail') ?: '';

            $categories = get_the_category();
            $category_list = array();
            foreach ($categories as $c) {
                $category_list[] = (int)$c->term_id;
            }

            if (in_array((int)$category_id, $category_list, true)) {
                $category_names = array();
                foreach ($categories as $c) {
                    $category_names[] = $c->name;
                }

                $posts[] = array(
                    'title'      => get_the_title(),
                    'url'        => get_permalink(),
                    'thumbnail'  => $thumbnail_url,
                    'categories' => $category_names,
                );
            }
        }
    }

    wp_reset_postdata();
    return $posts;
}

// 3. 再帰的にカテゴリを取得
function get_categories_recursive($parent_id = 0, $exclude_category_ids = array()) {

    // 開始カテゴリの取得（どっちのキーでも動くようにする）
    $start_category_id = get_option('caf_start_category', null);
    if (empty($start_category_id)) {
        $start_category_id = get_option('caf_start_category_field', null);
    }

    // 初回（parent_id=0）なら開始カテゴリを基準にする
    if (!is_null($start_category_id) && (int)$start_category_id > 0 && (int)$parent_id === 0) {
        $parent_id = (int)$start_category_id;
    }

    if (!is_array($exclude_category_ids)) {
        $exclude_category_ids = array();
    }

    $categories = get_terms(array(
        'taxonomy'   => 'category',
        'parent'     => (int)$parent_id,
        'hide_empty' => false,
    ));

    $category_data = array();

    if (!empty($categories) && !is_wp_error($categories)) {
        foreach ($categories as $category) {

            if (in_array((int)$category->term_id, $exclude_category_ids, true)) {
                continue;
            }

            $children = get_categories_recursive((int)$category->term_id, $exclude_category_ids);
            $posts    = get_posts_by_category((int)$category->term_id);

            if (!empty($children) || !empty($posts)) {
                $category_data[] = array(
                    'term_id'  => (int)$category->term_id,
                    'name'     => $category->name,
                    'slug'     => $category->slug,
                    'children' => $children,
                    'posts'    => $posts,
                );
            }
        }
    }

    return $category_data;
}

// 4. 親カテゴリの投稿を除外し、カテゴリ階層を表示（未使用なら消してOK）
function display_categories_with_posts($parent_id = 0) {
    $exclude_category_ids = get_all_parent_categories($parent_id);
    return get_categories_recursive($parent_id, $exclude_category_ids);
}

// 5. AJAXリクエストを処理する関数
function fetch_categories_callback() {

    // nonceチェック
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'caf_fetch_nonce')) {
        wp_send_json_error(array(
            'message' => 'Invalid nonce'
        ), 403);
        wp_die();
    }

    $parent_category_id = isset($_POST['parent_id']) ? (int) $_POST['parent_id'] : 0;
    $categories = get_categories_recursive($parent_category_id);

    if (!empty($categories)) {
        wp_send_json_success($categories);
    } else {
        wp_send_json_error(array(
            'message' => 'No categories/posts found (check start category option and excludes).',
            'parent_id' => $parent_category_id,
        ));
    }

    wp_die();
}


// 6. AJAXのアクションフックを登録
add_action('wp_ajax_fetch_categories', 'fetch_categories_callback');
add_action('wp_ajax_nopriv_fetch_categories', 'fetch_categories_callback');